// Email Verification System using EmailJS
// Handles assessment submission, magic link generation, and tool unlocking

const EMAIL_CONFIG = {
    serviceId: 'YOUR_EMAILJS_SERVICE_ID', // To be configured by user
    templateId: 'YOUR_EMAILJS_TEMPLATE_ID', // To be configured by user
    publicKey: 'YOUR_EMAILJS_PUBLIC_KEY' // To be configured by user
};

// Generate unique access token based on email and timestamp
function generateAccessToken(email) {
    const timestamp = Date.now();
    const randomPart = Math.random().toString(36).substring(2, 8).toUpperCase();
    const emailHash = btoa(email).substring(0, 8).toUpperCase();
    return `CE-${emailHash}-${randomPart}-${timestamp.toString(36).toUpperCase()}`;
}

// Send assessment results and verification email
async function sendAssessmentWithVerification(assessmentData) {
    try {
        // Generate unique access token
        const accessToken = generateAccessToken(assessmentData.email);
        
        // Create magic link
        const magicLink = `${window.location.origin}/verify.html?token=${accessToken}`;
        
        // Prepare email template parameters
        const templateParams = {
            to_email: assessmentData.email,
            to_name: assessmentData.firstName,
            from_name: 'CARIBEquity',
            magic_link: magicLink,
            access_token: accessToken,
            // Assessment summary
            total_score: assessmentData.totalScore,
            completion_date: new Date().toLocaleDateString(),
            // Full assessment data (for admin notification)
            assessment_json: JSON.stringify(assessmentData, null, 2)
        };
        
        // Send email using EmailJS
        const response = await emailjs.send(
            EMAIL_CONFIG.serviceId,
            EMAIL_CONFIG.templateId,
            templateParams,
            EMAIL_CONFIG.publicKey
        );
        
        if (response.status === 200) {
            // Store token temporarily (will be activated via magic link)
            sessionStorage.setItem('pending_token', accessToken);
            sessionStorage.setItem('pending_email', assessmentData.email);
            
            return {
                success: true,
                token: accessToken,
                message: 'Verification email sent successfully!'
            };
        } else {
            throw new Error('Failed to send email');
        }
        
    } catch (error) {
        console.error('Error sending verification email:', error);
        return {
            success: false,
            error: error.message || 'Failed to send verification email'
        };
    }
}

// Verify and activate access token from magic link
function activateAccessToken(token) {
    if (!token || token.length < 10) {
        return { success: false, error: 'Invalid token' };
    }
    
    // Store verified token in localStorage
    localStorage.setItem('caribequity_access_token', token);
    localStorage.setItem('caribequity_assessment_completed', 'true');
    localStorage.setItem('caribequity_verified_at', new Date().toISOString());
    
    // Clear pending session data
    sessionStorage.removeItem('pending_token');
    sessionStorage.removeItem('pending_email');
    
    return {
        success: true,
        message: 'Access verified! You can now use all tools.'
    };
}

// Check if user has verified access
function hasVerifiedAccess() {
    const token = localStorage.getItem('caribequity_access_token');
    const completed = localStorage.getItem('caribequity_assessment_completed');
    return token && completed === 'true';
}

// Get access token
function getAccessToken() {
    return localStorage.getItem('caribequity_access_token');
}

// Request new verification email (if user lost access)
async function resendVerificationEmail(email) {
    try {
        // Generate new token
        const accessToken = generateAccessToken(email);
        const magicLink = `${window.location.origin}/verify.html?token=${accessToken}`;
        
        const templateParams = {
            to_email: email,
            from_name: 'CARIBEquity',
            magic_link: magicLink,
            access_token: accessToken,
            subject: 'Your CARIBEquity Toolkit Access Link'
        };
        
        const response = await emailjs.send(
            EMAIL_CONFIG.serviceId,
            'resend_template_id', // Separate template for resend
            templateParams,
            EMAIL_CONFIG.publicKey
        );
        
        return {
            success: response.status === 200,
            message: 'Verification email resent successfully!'
        };
        
    } catch (error) {
        console.error('Error resending email:', error);
        return {
            success: false,
            error: 'Failed to resend verification email'
        };
    }
}

// Export functions for use in other scripts
window.EmailVerification = {
    sendAssessmentWithVerification,
    activateAccessToken,
    hasVerifiedAccess,
    getAccessToken,
    resendVerificationEmail,
    generateAccessToken
};
