/**
 * CARIBEquity Toolkit Tracking Library
 * Tracks tool usage and downloads, associating events with user onboarding data
 */

const CARIBEquityTracking = {
    API_URL: 'https://3001-i76y02u0pzgnuiqlwq2rw-632bd4ae.manusvm.computer/api',
    
    /**
     * Get user data from localStorage (set during onboarding)
     */
    getUserData() {
        const onboardingData = localStorage.getItem('caribequity_onboarding');
        if (!onboardingData) {
            return null;
        }
        try {
            return JSON.parse(onboardingData);
        } catch (e) {
            return null;
        }
    },

    /**
     * Track tool usage event
     */
    async trackUsage(toolName, eventType = 'page_view', metadata = {}) {
        const userData = this.getUserData();
        if (!userData || !userData.email) {
            console.warn('No user data found for tracking');
            return;
        }

        try {
            const response = await fetch(`${this.API_URL}/track/usage`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    userEmail: userData.email,
                    toolName,
                    eventType,
                    metadata
                })
            });

            if (!response.ok) {
                throw new Error('Failed to track usage');
            }

            console.log(`✅ Usage tracked: ${toolName} - ${eventType}`);
        } catch (error) {
            console.error('Error tracking usage:', error);
        }
    },

    /**
     * Track tool download event
     */
    async trackDownload(toolName) {
        const userData = this.getUserData();
        if (!userData || !userData.email) {
            console.warn('No user data found for tracking download');
            return;
        }

        try {
            const response = await fetch(`${this.API_URL}/track/download`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    userEmail: userData.email,
                    toolName
                })
            });

            if (!response.ok) {
                throw new Error('Failed to track download');
            }

            console.log(`✅ Download tracked: ${toolName}`);
        } catch (error) {
            console.error('Error tracking download:', error);
        }
    },

    /**
     * Initialize tracking for current page
     * Call this on page load
     */
    init(toolName) {
        // Track page view
        this.trackUsage(toolName, 'page_view');

        // Track page unload (session end)
        const startTime = Date.now();
        window.addEventListener('beforeunload', () => {
            const sessionDuration = Math.floor((Date.now() - startTime) / 1000);
            this.trackUsage(toolName, 'session_end', { duration: sessionDuration });
        });
    }
};

// Auto-initialize if tool name is set in meta tag
document.addEventListener('DOMContentLoaded', () => {
    const toolMeta = document.querySelector('meta[name="tool-name"]');
    if (toolMeta) {
        const toolName = toolMeta.content;
        CARIBEquityTracking.init(toolName);
    }
});
